/**********************************************************
/**
* @file	internalClockAdjust
*
* This file contains an example to adjust internal clock freq
*
*
* MODIFICATION HISTORY:
*
* Ver   Who		Date      Changes
* ---   ------	--------  --------------------------------
* 1.0	yuchun  02/22/17  First release
*
***********************************************************/
#include "MCU.h"
#include "GPIO.h"
#include "CORE.h"
#include "internalClockAdjust.h"

/*********************************************************/
/**
* This function is the main function
*
*
**********************************************************/

//this code adjust internal osc to the target freq base on the measured freq on pin CLK_375k
//the target freq is only approximate, due to the discrete property of the setting parameter
//the final osc freq can only be close to the target freq, 
// here is the how the adj is determined for a target osc freq (target_osc_freq) and a measurend current osc freq (measure_osc_freq)
// if target_osc_freq  < measure_osc_freq:
//    target_osc_freq =~ measure_osc_freq * (1-0.05)^m , where m is the closest integer to make equation valid
//    adj = -m
// else if target_osc_freq > measure_osc_freq:
//    target_osc_freq =~ measure_osc_freq * (1 + 0.05)^m, where m is the closest integer to make equation valid
//    adj = m
// adj must be in range [-8, 7]
void calibOSC(int32_t adj)
{
    if (adj < 0) {
        if (adj < -8)
            adj = -8; //max adjustment allowed
         adj = 7-adj;
    } else {
        if (adj > 7)
            adj = 7;
    }
    MCU_REG_WRITE(MCU_PLL_CTRL, OSC_ADJ_Msk, OSC_ADJ_Pos, adj);
}

//adjust PLL freq to adj * osc_freq, osc_freq is the internal osc freq or external osc freq if crystal is used
// adj must be even number in range [4, 32]
void adjPLL(uint32_t adj)
{
    register uint32_t n, pll_bits; 
    if (adj < 4)
        adj = 4; 
    else if (adj > 32)
        adj = 32;
    n = (adj>>1) - 1;
    switch (n) {
    case 0 :
        //not allowed
        break;
    case 1 :
        pll_bits = 0x4;
        break;
    case 2 :
        pll_bits = 0x7;
        break;
    case 3 :
        pll_bits = 0x6;
        break;
    case 4 :
        pll_bits = 0x1;
        break;
    case 5 :
        pll_bits = 0x0;
        break;
    case 6 :
        pll_bits = 0x3;
        break;
    case 7 :
        pll_bits = 0x2;
        break;
    case 8 :
        pll_bits = 0xd;
        break;
    case 9 :
        pll_bits = 0xc;
        break;
    case 10 :
        pll_bits = 0xf;
        break;
    case 11 :
        pll_bits = 0xe;
        break;
    case 12 :
        pll_bits = 0x9;
        break;
    case 13 :
        pll_bits = 0x8;
        break;
    case 14 :
        pll_bits = 0xb;
        break;
    case 15 :
        pll_bits = 0xa;
        break;
    default :
        break;
    }
    MCU_REG_WRITE(MCU_PLL_CTRL, PLL_REFDIV_Msk, PLL_REFDIV_Pos, pll_bits);
}

//adjust MCU main clock freq with respect to PLL freq, 
// the main clock freq will be mcu_freq = pll_freq/adj
// the adj can take these values [2, 4, 8, 16]
void adjMCUClk(uint32_t adj)
{
    register uint32_t clk_bit;
    switch (adj) {
    case 2 :
        clk_bit = 0x3;
        break;
    case 4 :
        clk_bit = 0x2;
        break;
    case 8 :
        clk_bit = 0x1;
        break;
    case 16 :
        clk_bit = 0x0;
        break;
    default:
        clk_bit = 0x00;
        break;
    }
    MCU_REG_WRITE(MCU_CLK_CTRL, CLK_DIV_Msk, CLK_DIV_Pos, clk_bit);
}

//this code adjust osc freq, PLL freq, and MCU freq
void adjustMCUClock(int32_t oscAdj, uint32_t pllAdj, uint32_t clkAdj)
{
	//turn off system debug, enable all GPIO
    MCU_REG_WRITE(MCU_CTRL, CTRL_GPIO_EN_Msk, CTRL_GPIO_EN_Pos, 1);
    calibOSC(oscAdj);
    adjPLL(pllAdj);
    adjMCUClk(clkAdj);
    RT_Delay_ms(10);  //delay 10ms for system to stable after clock adjust
}









































































































































































































